﻿using System.Drawing;
using Onitama.Core.PlayerAggregate;
using Onitama.Core.PlayerAggregate.Contracts;
using Onitama.Core.TableAggregate.Contracts;
using Onitama.Core.UserAggregate;
using Onitama.Core.Util;

namespace Onitama.Core.TableAggregate;

/// <inheritdoc cref="ITable"/>
internal class Table : ITable
{
    private static readonly Color[] PossibleColors =
        new[] { Color.Red, Color.Blue, Color.Green, Color.Yellow, Color.Orange };
    
    
    private Guid _id;
    private TablePreferences _preferences;
    private Guid _ownerPlayerId;
    private readonly List<IPlayer> _seatedPlayers;
    private bool _hasAvailableSeat;
    private Guid _gameId;

    public Table(Guid id, TablePreferences preferences)
    {
        _id = id;
        _preferences = preferences;
        _hasAvailableSeat = true;
        _seatedPlayers = new List<IPlayer>();
    }

    public Guid Id
    {
        get { return _id; }
    }

    public TablePreferences Preferences
    {
        get { return _preferences; }
    }

    public Guid OwnerPlayerId
    {
        get { return _ownerPlayerId; }
    }

    public IReadOnlyList<IPlayer> SeatedPlayers => _seatedPlayers;

    public bool HasAvailableSeat
    {
        get { return _hasAvailableSeat; }
    }

    public Guid GameId
    {
        get { return _gameId; }
        set { _gameId = value; }
    }

    public void Join(User user)
    {
        Random random = new Random();


        _seatedPlayers.ForEach(player =>
        {
            if (player.Id == user.Id)
            {
                throw new InvalidOperationException("User is already seated");
            }
        });

        if (!_hasAvailableSeat)
        {
            throw new InvalidOperationException("The table is full");
        }

        if (SeatedPlayers.Count == 0)
        {
            _ownerPlayerId = user.Id;
        }

        Color playerColor = PossibleColors[random.Next(0, PossibleColors.Length)];
        Direction direction = Direction.North;
        

        if (SeatedPlayers.Count == 1)
        {
            while (playerColor == SeatedPlayers[0].Color)
            {
                playerColor = PossibleColors[random.Next(0, PossibleColors.Length)];
            }
            
            if (SeatedPlayers[0].Direction == Direction.North)
            {
                direction = Direction.South;
            }
        }

        _seatedPlayers.Add(new HumanPlayer(user.Id, user.WarriorName, playerColor, direction));


        if (_seatedPlayers.Count == Preferences.NumberOfPlayers)
        {
            _hasAvailableSeat = false;
        }
    }

    public void Leave(Guid userId)
    {
        IPlayer playerToRemove = null;
        // Reverse loop to prevent IndexOutOfRange
        _seatedPlayers.ForEach(player =>
        {
            if (player.Id == userId)
            {
                playerToRemove = player;
            }
        });

        if (playerToRemove == null)
        {
            throw new InvalidOperationException("User is not part of the table");
        }
        else
        {
            _seatedPlayers.Remove(playerToRemove);
        }

        if (SeatedPlayers.Count == 1)
        {
            _ownerPlayerId = SeatedPlayers[0].Id;
        }

        _hasAvailableSeat = true;
    }

    public void FillWithArtificialPlayers(IGamePlayStrategy gamePlayStrategy)
    {
    }
}