﻿using Onitama.Core.GameAggregate;
using Onitama.Core.GameAggregate.Contracts;
using Onitama.Core.MoveCardAggregate.Contracts;
using Onitama.Core.PlayMatAggregate.Contracts;
using Onitama.Core.SchoolAggregate.Contracts;
using Onitama.Core.Util;
using Onitama.Core.Util.Contracts;
using Onitama.Core.PlayerAggregate.Contracts;

namespace Onitama.Core.PlayMatAggregate;

/// <inheritdoc cref="IPlayMat"/>
internal class PlayMat : IPlayMat
{
    private int _size;
    private IPawn[,] _grid;

    /// <summary>
    /// Creates a play mat that is a copy of another play mat
    /// </summary>
    /// <param name="otherPlayMat">The play mat to copy</param>
    /// <param name="copiedPlayers">
    /// Copies of the players (with their school)
    /// that should be used to position pawn on the copy of the <paramref name="otherPlayMat"/>.</param>
    /// <remarks>
    /// This is an EXTRA. Not needed to implement the minimal requirements.
    /// To make the mini-max algorithm for an AI game play strategy work, this constructor should be implemented.
    /// </remarks>
    public PlayMat(IPlayMat otherPlayMat, IPlayer[] copiedPlayers)
    {
        throw new NotImplementedException("TODO: copy properties of other playmat");
    }

    public PlayMat(int size)
    {
        _size = size;
        _grid = new IPawn[size, size];
    }

    public IPawn[,] Grid => _grid;
    public int Size => _size;

    public void PositionSchoolOfPlayer(IPlayer player)
    {
        Direction playerDirection = player.Direction;
        IPawn[] playerAllPawns = player.School.AllPawns;

        ICoordinate startCoordinate = playerDirection.GetStartCoordinate();

        // Master will always be in the middle of array
        for (int column = 0; column < playerAllPawns.Length; column++)
        {
            IPawn pawn = playerAllPawns[column];
            // North or South means row is the same as startCoordinate
            if (playerDirection == Direction.North || playerDirection == Direction.South)
            {
                int row = startCoordinate.Row;
                Coordinate coordinate = new Coordinate(row, column);
                _grid[row, column] = pawn;
                pawn.Position = coordinate;
            }
        }
    }

    public IReadOnlyList<IMove> GetValidMoves(IPawn pawn, IMoveCard card, Direction playerDirection)
    {
        IList<IMove> possibleMoves = new List<IMove>();
        IReadOnlyList<ICoordinate> possibleTargets =
            card.GetPossibleTargetCoordinates(pawn.Position, playerDirection, Size);
        foreach (ICoordinate possibleTarget in possibleTargets)
        {
            IPawn target = Grid[possibleTarget.Row, possibleTarget.Column];
            if (target == null || (target.OwnerId != pawn.OwnerId))
            {
                IMove move = new Move(card, pawn, playerDirection, possibleTarget);
                possibleMoves.Add(move);
            }
        }

        return possibleMoves.AsReadOnly();
    }

    public void ExecuteMove(IMove move, out IPawn capturedPawn)
    {
        IReadOnlyList<IMove> possibleMoves = GetValidMoves(move.Pawn, move.Card, move.PlayerDirection);
        if (!possibleMoves.Contains(move))
        {
            throw new ApplicationException("This is not a valid move");
        }

        capturedPawn = Grid[move.To.Row, move.To.Column];
        Grid[move.To.Row, move.To.Column] = move.Pawn;
        Grid[move.Pawn.Position.Row, move.Pawn.Position.Column] = null;
        move.Pawn.Position = move.To;
    }
}