import { getUserId } from "./customAuthFlow.js";
import { getBearer } from "./customBearer.js";
import { customFetch } from "./customFetch.js";
import { createNotification } from "./customNotification.js";
import { makePageWinning } from "./endScreenWinning.js";
import { makePageLosing } from "./endScreenLosing.js";
import { showDefeatedPawn } from "./customDefeatedPawn.js";

// BOARD AND PANWS
function createMaster(player, id) {
  let color = player.color.toLowerCase();
  let div = document.createElement("div");
  div.classList.add(
    "w-full",
    "h-full",
    "flex",
    "justify-center",
    "items-center"
  );
  let img = document.createElement("img");
  img.src = `/Frontend/src/assets/images/king.png`;
  img.classList.add(
    `drop-shadow-pawn-${color}`,
    "translate-y-pawn",
    "max-w-full",
    "max-h-full",
    "w-3/5"
  );
  img.id = id;
  div.appendChild(img);
  return div;
}

function createStudent(player, id) {
  let color = player.color.toLowerCase();
  let div = document.createElement("div");
  div.classList.add(
    "w-full",
    "h-full",
    "flex",
    "justify-center",
    "items-center"
  );
  let img = document.createElement("img");
  img.src = `/Frontend/src/assets/images/pawn.png`;
  img.classList.add(
    `drop-shadow-pawn-${color}`,
    "translate-y-pawn",
    "max-w-full",
    "max-h-full",
    "w-3/4"
  );
  img.id = id;
  div.appendChild(img);
  return div;
}

function createPawn(player, pawn) {
  // Master pawn
  if (pawn.type == 0) {
    return createMaster(player, pawn.id);
  }
  return createStudent(player, pawn.id);
}

function displayPawn(player, pawn) {
  let createdPawn = createPawn(player, pawn);
  let square = document.getElementById(
    `${pawn.position.row},${pawn.position.column}`
  );
  square.replaceChildren();
  square.appendChild(createdPawn);
  createdPawn.addEventListener("click", () => selectPawn(pawn));
}

function createSchool(player) {
  player.school.allPawns.forEach((pawn) => {
    displayPawn(player, pawn);
  });
}

// Counts the amount of pawns
function countPawns(playMat) {
  if (!playMat || !playMat.grid) {
    return 0;
  }

  let count = 0;
  playMat.grid.forEach((row) => {
    row.forEach((column) => {
      if (column) {
        count++;
      }
    });
  });
  return count;
}

function createSchoolArray(player) {
  return [
    createStudent(player),
    createStudent(player),
    createMaster(player),
    createStudent(player),
    createStudent(player),
  ];
}

function removeSchool(player) {
  let row = player.direction == "North" ? 0 : 4;
  let column = 0;
  let school = createSchoolArray(player);
  school.forEach((pawn) => {
    let square = document.getElementById(`${row},${column}`);
    square.replaceChildren();
    column++;
  });
}

function createBoard(playmatSize = 5) {
  let board = [];
  for (let row = 0; row < playmatSize; row++) {
    let rowArray = [];
    for (let column = 0; column < playmatSize; column++) {
      let div = document.createElement("div");
      div.classList.add(
        "w-1/5",
        "h-1/5",
        "flex",
        "justify-center",
        "items-center"
      );
      div.id = `${row},${column}`;
      // Showing ID div.innerHTML = `${row},${column}`;
      let moveTo = {
        row,
        column,
      };
      div.addEventListener("click", () => makeMoveTo(moveTo));
      if ((column + row) % 2 == 0) {
        div.classList.add("bg-gray-700");
      } else {
        div.classList.add("bg-zinc-800");
      }
      rowArray.push(div);
    }
    board.unshift(rowArray);
  }
  return board;
}

function rotateItem(item, direction = "North") {
  if (direction == "South") {
    let reversedBoard = [];
    for (let row = item.length - 1; row >= 0; row--) {
      let rowArray = [];
      for (let column = item[row].length - 1; column >= 0; column--) {
        rowArray.push(item[row][column]);
      }
      reversedBoard.push(rowArray);
    }
    return reversedBoard;
  }
  return item;
}

function displayBoard(direction) {
  let board = createBoard();
  board = rotateItem(board, direction);
  let boardDiv = document.getElementById("board");
  boardDiv.classList.add(
    "border-2",
    "border-black",
    "w-full",
    "md:w-auto",
    "md:max-h-[55%]",
    "aspect-square",
    "flex",
    "flex-wrap"
  );
  boardDiv.id = "board";

  board.forEach((row) => {
    row.forEach((column) => {
      boardDiv.appendChild(column);
    });
  });
}

function clearBoard() {
  let board = document.getElementById("board");
  let squares = board.getElementsByTagName("div");
  for (const square of squares) {
    square.replaceChildren();
  }
}

function  createCard(name, grid, color) {
  const card = document.createElement("div");
  card.classList.add(
    "w-40",
    "h-30",
    "rounded-lg",
    "flex",
    "flex-row",
    "items-center",
    "justify-between",
    "border-2",
    "border-black"
  );
  card.style.backgroundImage =
    'url("/Frontend/src/assets/images/cardBackground.png")';
  card.style.backgroundSize = "cover";

  // inner div
  const cardDiv = document.createElement("div");
  cardDiv.classList.add("w-1/2", "flex", "flex-col", "items-center");

  // photo animal
  const imgAnimal = document.createElement("img");
  imgAnimal.setAttribute(
    "src",
    `/Frontend/src/assets/images/cardImages/${name}.png`
  );
  imgAnimal.classList.add("w-9", "h-9");
  cardDiv.appendChild(imgAnimal);

  // name div
  const nameLabel = document.createElement("div");
  nameLabel.textContent = name;
  nameLabel.classList.add("text-base", "font-extrabold");
  cardDiv.appendChild(nameLabel);

  //grid
  const gridContainer = document.createElement("div");
  gridContainer.classList.add("grid", "grid-cols-5", "w-1/2", "h-20", "p-1");

  for (let i = 4; i >= 0; i--) {
    for (let j = 0; j < 5; j++) {
      const cell = document.createElement("div");
      cell.classList.add("border", `border-black`, "grid-cell");

      const value = grid[i][j];
      if (value == "1") {
        cell.classList.add(
          `bg-${color.toLowerCase()}-${
            color.toLowerCase() == "yellow" ? 400 : 500
          }`
        );
      } else if (value == "2") {
        cell.classList.add("bg-gray-200");
      } else {
        cell.classList.add("bg-black", "bg-opacity-10");
      }

      gridContainer.appendChild(cell);
    }
  }

  card.id = name;
  card.appendChild(cardDiv);
  card.appendChild(gridContainer);

  return card;
}

function displayCard(element, card, direction) {
  let rotatedGrid = rotateItem(card.grid, direction);
  let createdCard = createCard(card.name, rotatedGrid, card.stampColor);
  createdCard.addEventListener("click", () => selectCard(card));
  element.appendChild(createdCard);
}

// GAME FUNCTIONS

async function startGame(tableId) {
  let response = await customFetch(
    `api/Tables/${tableId}/start-game`,
    "POST",
    null,
    getBearer()
  );

  if (!response.ok) {
    createNotification(response.message);
  } else {
    createNotification("Game starting soon");
  }
}

async function getGame(id) {
  let response = await customFetch(`api/Games/${id}`, "GET", null, getBearer());

  if (!response.ok) {
    gameNotFound(id);
  } else {
    let game = response.data;
    if (!game.players.some((player) => player.id == getUserId())) {
      window.location.href =
        "/Frontend/src/auth/lobby.html?notification=You're not part of this game";
    }
    return game;
  }
}

// Everything that can change
let enemyPlayerDisplayed = {};
let currentPlayerDisplayed = {};
let enemyPlayerMoveCardsDisplayed = [];
let currentPlayerMoveCardsDisplayed = [];
let currentPlayMatDisplayed = {};
let playerToPlayIdDisplayed;
let selectedPawn;
let selectedCard;
let currentGameId;
let possibleMovesDisplayed;

async function displayGame(gameId) {
  let game = await getGame(gameId);
  currentGameId = game.id;

  let currentPlayer;
  let enemyPlayer;

  if (game.players[0].id == getUserId()) {
    currentPlayer = game.players[0];
    enemyPlayer = game.players[1];
  } else {
    currentPlayer = game.players[1];
    enemyPlayer = game.players[0];
  }

  let currentPlayerMoveCards = currentPlayer.moveCards;
  let enemyPlayerMoveCards = enemyPlayer.moveCards;

  // Display current player
  if (
    Object.keys(currentPlayerDisplayed).length == 0 ||
    compareJson(currentPlayer, currentPlayerDisplayed)
  ) {
    let currentPlayerName = document
      .getElementById("currentPlayer")
      .querySelector("h1");
    updatePlayerDisplay(currentPlayerName, currentPlayer);
    currentPlayerDisplayed = currentPlayer;
  }

  // Display enemy player
  if (
    Object.keys(enemyPlayerDisplayed).length == 0 ||
    compareJson(enemyPlayer, enemyPlayerDisplayed)
  ) {
    let enemyPlayerName = document
      .getElementById("enemyPlayer")
      .querySelector("h1");
    updatePlayerDisplay(enemyPlayerName, enemyPlayer);
    enemyPlayerDisplayed = enemyPlayer;
  }

  // Display current player cards
  if (compareJson(currentPlayerMoveCards, currentPlayerMoveCardsDisplayed)) {
    let element = document.getElementById("currentPlayerCards");
    element.replaceChildren();
    currentPlayerMoveCardsDisplayed = [];
    updateCardsDisplay(element, currentPlayerMoveCards)
    currentPlayerMoveCardsDisplayed = currentPlayerMoveCards;
  }

  // Display enemy player cards
  if (compareJson(enemyPlayerMoveCards, enemyPlayerMoveCardsDisplayed)) {
    let element = document.getElementById("enemyPlayerCards");
    element.replaceChildren();
    enemyPlayerMoveCardsDisplayed = [];
    updateCardsDisplay(element, enemyPlayerMoveCards, "South")
    enemyPlayerMoveCardsDisplayed = enemyPlayerMoveCards;
  }

  if (compareJson(game.playMat, currentPlayMatDisplayed)) {
    clearBoard();
    // Display board
    game.playMat.grid.forEach((row) => {
      row.forEach((pawn) => {
        if (pawn) {
          let playerOwner;
          if (pawn.ownerId == enemyPlayer.id) {
            playerOwner = enemyPlayer;
          } else {
            playerOwner = currentPlayer;
          }
          displayPawn(playerOwner, pawn);
        }
      });
    });
    if (countPawns(game.playMat) < countPawns(currentPlayMatDisplayed)) {
      showDefeatedPawn();
    }
    currentPlayMatDisplayed = game.playMat;
  }

  // Set player to play
  let currentPlayerDiv = document.getElementById("currentPlayer");
  let enemyPlayerDiv = document.getElementById("enemyPlayer");
  if (game.playerToPlayId == currentPlayer.id) {
    currentPlayerDiv.classList.add("bg-yellow-500");
    enemyPlayerDiv.classList.remove("bg-yellow-500");
  } else {
    currentPlayerDiv.classList.remove("bg-yellow-500");
    enemyPlayerDiv.classList.add("bg-yellow-500");
  }

  // Display extra card
  if (playerToPlayIdDisplayed != game.playerToPlayId) {
    let currentPlayerExtraCardDiv = document.getElementById(
      "currentPlayerExtraCard"
    );
    currentPlayerExtraCardDiv.replaceChildren();
    let enemyPlayerExtraCardDiv = document.getElementById(
      "enemyPlayerExtraCard"
    );
    enemyPlayerExtraCardDiv.replaceChildren();
    // Current player at turn
    if (currentPlayer.id == game.playerToPlayId) {
      displayCard(currentPlayerExtraCardDiv, game.extraMoveCard, "North");
    }
    // Enemy player at turn
    else {
      displayCard(enemyPlayerExtraCardDiv, game.extraMoveCard, "South");
    }
    playerToPlayIdDisplayed = game.playerToPlayId;
  }

  if (game.winnerPlayerId != "00000000-0000-0000-0000-000000000000") {
    document.body.innerHTML = "";
    const winningMethod = game.winnerMethod;
    if (game.winnerPlayerId == currentPlayer.id) {
      makePageWinning(winningMethod);
    } else {
      makePageLosing();
    }
  } else {
    setTimeout(() => {
      displayGame(game.id);
    }, 5000);
  }
}

function gameNotFound(id) {
  window.location.href = `/Frontend/src/auth/lobby.html?notification=Game with id "${id}" not found`;
}

function updatePlayerDisplay(element, player) {
  element.replaceChildren();
  let i = document.createElement("i");
  i.classList.add("fa-solid", "fa-square");
  i.style.color = player.color;
  element.appendChild(i);
  element.appendChild(document.createTextNode(player.name));
}

function updateCardsDisplay(element, cards, direction = "North"){
  cards.forEach(async (card) => {
    displayCard(element, card, direction);
  });
}

// GAME MECHANICS

function selectPawn(pawn) {
  if (isPlayerAtTurn(currentPlayerDisplayed)) {
    createNotification("Player is not at turn");
  } else if (pawn.ownerId == currentPlayerDisplayed.id) {
    // Remove previous border
    if (selectedPawn) {
      let selectedPawnDiv = document.getElementById(
        `${selectedPawn.position.row},${selectedPawn.position.column}`
      );
      selectedPawnDiv.classList.remove("!bg-gray-900");
    }

    // Deselect
    if (JSON.stringify(selectedPawn) == JSON.stringify(pawn)) {
      selectedPawn = undefined;
    } else {
      selectedPawn = pawn;
      let newSelectedPawnDiv = document.getElementById(
        `${selectedPawn.position.row},${selectedPawn.position.column}`
      );
      newSelectedPawnDiv.classList.add("!bg-gray-900");
    }
    showPossibleMoves();
  }
}

function selectCard(card) {
  if (isPlayerAtTurn(currentPlayerDisplayed)) {
    createNotification("Player is not at turn");
  } else if (
    !JSON.stringify(currentPlayerDisplayed.moveCards).includes(
      JSON.stringify(card)
    )
  ) {
    createNotification("This is not your card");
  } else {
    // Remove previous border
    if (selectedCard) {
      let selectedCardDiv = document.getElementById(`${selectedCard.name}`);
      selectedCardDiv.classList.remove("border-white");
    }

    // Deselect
    if (selectedCard == card) {
      selectedCard = undefined;
    } else {
      selectedCard = card;
      let newSelectedCardDiv = document.getElementById(`${selectedCard.name}`);
      newSelectedCardDiv.classList.add("border-white");
    }
    showPossibleMoves();
  }
}

function isPlayerAtTurn(player) {
  return playerToPlayIdDisplayed != player.id;
}

async function showPossibleMoves() {
  if (possibleMovesDisplayed) {
    possibleMovesDisplayed.forEach((move) => {
      let moveTo = `${move.to.row},${move.to.column}`;
      let square = document.getElementById(moveTo);
      square.classList.remove("border-2");
    });
  }

  if (selectedCard && selectedPawn) {
    let response = await customFetch(
      `api/Games/${currentGameId}/possible-moves-of/${selectedPawn.id}/for-card/${selectedCard.name}`,
      "GET",
      null,
      getBearer()
    );

    if (!response.ok) {
      createNotification(response.status);
    } else {
      let possibleMoves = response.data;

      if (possibleMoves != possibleMovesDisplayed) {
        possibleMovesDisplayed = possibleMoves;

        possibleMoves.forEach((move) => {
          let moveTo = `${move.to.row},${move.to.column}`;
          let square = document.getElementById(moveTo);
          square.classList.add("border-2");
        });
      }
    }
  }
}

async function makeMoveTo(moveTo) {
  if (
    selectedCard &&
    selectedPawn &&
    JSON.stringify(selectedPawn.position) != JSON.stringify(moveTo)
  ) {
    let body = {
      pawnId: selectedPawn.id,
      moveCardName: selectedCard.name,
      to: {
        row: moveTo.row,
        column: moveTo.column,
      },
    };
    let response = await customFetch(
      `api/Games/${currentGameId}/move-pawn`,
      "POST",
      body,
      getBearer()
    );
    if (!response.ok) {
      createNotification(response.message);
    } else {
      selectCard(selectedCard);
      selectPawn(selectedPawn);
      showPossibleMoves();
    }
    displayGame(currentGameId);
  }
}

async function getDirectionForPlayerGame(gameId) {
  await getGame(gameId);
  let response = await customFetch(
    `api/Games/${gameId}`,
    "GET",
    null,
    getBearer()
  );

  if (!response.ok) {
    gameNotFound(id);
  } else {
    let game = response.data;
    const player = game.players.find((player) => player.id == getUserId());
    if (player) {
      return player.direction;
    }
    return "North";
  }
}

async function skipMovement() {
  if (isPlayerAtTurn(currentPlayerDisplayed)) {
    createNotification("Player is not at turn");
  } else if (!selectedCard) {
    createNotification("Please select a card to skip");
  } else {
    let body = {
      moveCardName: selectedCard.name,
    };
    let response = await customFetch(
      `api/Games/${currentGameId}/skip-movement`,
      "POST",
      body,
      getBearer()
    );
    if (!response.ok) {
      createNotification(response.message);
    } else {
      displayGame(currentGameId);
    }
  }
}

// EXTRA
function compareJson(object1, object2) {
  return JSON.stringify(object1) != JSON.stringify(object2);
}

export {
  createBoard,
  createSchool,
  clearBoard,
  rotateItem,
  displayBoard,
  createCard,
  startGame,
  removeSchool,
  getGame,
  displayGame,
  gameNotFound,
  skipMovement,
  getDirectionForPlayerGame,
};
