﻿using Onitama.Core.GameAggregate.Contracts;
using Onitama.Core.MoveCardAggregate.Contracts;
using Onitama.Core.PlayerAggregate.Contracts;
using Onitama.Core.PlayMatAggregate.Contracts;
using Onitama.Core.SchoolAggregate.Contracts;
using Onitama.Core.Util.Contracts;

namespace Onitama.Core.GameAggregate;

/// <inheritdoc cref="IGame"/>
internal class Game : IGame
{
    /// <summary>
    /// Creates a new game and determines the player to play first.
    /// </summary>
    /// <param name="id">The unique identifier of the game</param>
    /// <param name="playMat">
    /// The play mat
    /// (with the schools of the player already positioned on it)
    /// </param>
    /// <param name="players">
    /// The 2 players that will play the game
    /// (with 2 move cards each)
    /// </param>
    /// <param name="extraMoveCard">
    /// The fifth card used to exchange cards after the first move
    /// </param>
    public Game(Guid id, IPlayMat playMat, IPlayer[] players, IMoveCard extraMoveCard)
    {
        Id = id;
        PlayMat = playMat;
        Players = players;
        _extraMoveCard = extraMoveCard;
        foreach (IPlayer player in players)
        {
            playMat.PositionSchoolOfPlayer(player);
            if (extraMoveCard.StampColor == player.Color)
            {
                _playerToPlayId = player.Id;
            }
        }
    }

    /// <summary>
    /// Creates a game that is a copy of another game.
    /// </summary>
    /// <remarks>
    /// This is an EXTRA. Not needed to implement the minimal requirements.
    /// To make the mini-max algorithm for an AI game play strategy work, this constructor should be implemented.
    /// </remarks>
    public Game(IGame otherGame)
    {
        throw new NotImplementedException("TODO: copy the properties of the other game");
        //Attention: the players should be copied, not just referenced
    }

    private Guid _playerToPlayId;
    private Guid _winnerPlayerId;
    private string _winnerMethod;
    private IMoveCard _extraMoveCard;

    public Guid Id { get; }
    public IPlayMat PlayMat { get; }
    public IMoveCard ExtraMoveCard => _extraMoveCard;
    public IPlayer[] Players { get; }
    public Guid PlayerToPlayId => _playerToPlayId;
    public Guid WinnerPlayerId => _winnerPlayerId;
    public string WinnerMethod => _winnerMethod;
    public IReadOnlyList<IMove> GetPossibleMovesForPawn(Guid playerId, Guid pawnId, string moveCardName)
    {
        // Get the player
        IPlayer player = GetPlayerById(playerId);
        
        // Get the pawn
        IPawn pawn = GetPawnFromPlayer(player, pawnId);
        
        // Get the card
        IMoveCard card = GetCardFromPlayer(player, moveCardName);
        
        return PlayMat.GetValidMoves(pawn, card, player.Direction);
    }

    public IReadOnlyList<IMove> GetAllPossibleMovesFor(Guid playerId)
    {
        IList<IMove> possibleMoves = new List<IMove>();

        IPlayer player = GetPlayerById(playerId);
        
        foreach (IMoveCard playerMoveCard in player.MoveCards)
        {
            foreach (IPawn pawn in player.School.AllPawns)
            {
                foreach (IMove move in PlayMat.GetValidMoves(pawn, playerMoveCard, player.Direction))
                {
                    possibleMoves.Add(move);
                }
            }
        }

        return possibleMoves.AsReadOnly();
    }

    public void MovePawn(Guid playerId, Guid pawnId, string moveCardName, ICoordinate to)
    {
        if (playerId != PlayerToPlayId)
        {
            throw new ApplicationException($"The player with id '{playerId}' is not at turn");
        }

        IPlayer player = GetPlayerById(playerId);
        IPawn pawn = GetPawnFromPlayer(player, pawnId);
        IMoveCard card = GetCardFromPlayer(player, moveCardName);
        IMove move = new Move(card, pawn, player.Direction, to);

        IPawn capturedPawn = null;
        PlayMat.ExecuteMove(move, out capturedPawn);

        if (capturedPawn is { Type: PawnType.Master })
        {
            _winnerPlayerId = player.Id;
            _winnerMethod = "Way Of The Stone";
        }

        IPlayer enemyPlayer = GetNextOpponent(player.Id);
        if (Equals(to, enemyPlayer.School.TempleArchPosition))
        {
            _winnerPlayerId = player.Id;
            _winnerMethod = "Way Of The Stream";
        }
        
        ChangePlayerCard(player, card);
        ChangeNextPlayer(player);
    }

    public void SkipMovementAndExchangeCard(Guid playerId, string moveCardName)
    {
        if (PlayerToPlayId != playerId)
        {
            throw new ApplicationException($"The player with id '{playerId}' is not at turn");
        }
        
        IPlayer player = GetPlayerById(playerId);

        if (GetAllPossibleMovesFor(player.Id).Count != 0)
        {
            throw new ApplicationException("The player does have valid moves");
        }
        
        IMoveCard card = GetCardFromPlayer(player, moveCardName);
        
        ChangePlayerCard(player, card);
        ChangeNextPlayer(player);
    }

    public IPlayer GetNextOpponent(Guid playerId)
    {
        IPlayer player = GetPlayerById(playerId);
        int nextIndex = (Array.IndexOf(Players, player) + 1) % Players.Length;
        return Players[nextIndex];
    }

    public void ChangePlayerCard(IPlayer player, IMoveCard card)
    {
        player.MoveCards.Remove(card);
        player.MoveCards.Add(_extraMoveCard);
        _extraMoveCard = card;
    }

    public void ChangeNextPlayer(IPlayer currentPlayer)
    {
        IPlayer nextPlayer = GetNextOpponent(currentPlayer.Id);
        _playerToPlayId = nextPlayer.Id;
    }

    public IPlayer GetPlayerById(Guid playerId)
    {
        IPlayer player = null;
        
        foreach (IPlayer seatedPlayer in Players)
        {
            if (seatedPlayer.Id == playerId)
            {
                player = seatedPlayer;
            }
        }

        if (player == null)
        {
            throw new InvalidOperationException($"The player with {playerId} doesn't exist");
        }

        return player;
    }

    public IPawn GetPawnFromPlayer(IPlayer player, Guid pawnId)
    {
        return player.School.GetPawn(pawnId);
    }

    public IMoveCard GetCardFromPlayer(IPlayer player, string moveCardName)
    {
        IMoveCard card = null;
        foreach (IMoveCard moveCard in player.MoveCards)
        {
            if (moveCard.Name == moveCardName)
            {
                card = moveCard;
            }
        }

        if (card == null)
        {
            throw new ApplicationException($"The card with name '{moveCardName}' doesn't exist on player '{player.Name}'");
        }

        return card;
    }
}