/*
 * Decompiled with CFR 0.152.
 */
package greymerk.roguelike.dungeon;

import com.github.fnar.minecraft.block.BlockType;
import com.github.fnar.util.ReportThisIssueException;
import com.github.fnar.util.TimedTask;
import greymerk.roguelike.config.RogueConfig;
import greymerk.roguelike.dungeon.DungeonLevel;
import greymerk.roguelike.dungeon.DungeonStage;
import greymerk.roguelike.dungeon.settings.DungeonSettings;
import greymerk.roguelike.dungeon.settings.SettingsRandom;
import greymerk.roguelike.dungeon.settings.SettingsResolver;
import greymerk.roguelike.dungeon.settings.SpawnCriteria;
import greymerk.roguelike.dungeon.tasks.DungeonTaskRegistry;
import greymerk.roguelike.dungeon.tasks.IDungeonTask;
import greymerk.roguelike.dungeon.towers.TowerType;
import greymerk.roguelike.theme.Theme;
import greymerk.roguelike.worldgen.Coord;
import greymerk.roguelike.worldgen.VanillaStructure;
import greymerk.roguelike.worldgen.WorldEditor;
import greymerk.roguelike.worldgen.filter.Filter;
import greymerk.roguelike.worldgen.shapes.RectSolid;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.IntStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class Dungeon {
    public static final int NUM_LAYERS = 5;
    public static final int VERTICAL_SPACING = 10;
    public static final int TOPLEVEL = 50;
    public static final int CHUNK_SIZE = 16;
    public static final int BOTTOM_OF_WORLD_HEIGHT = 5;
    public static final String MOD_ID = "roguelike";
    private static final Logger logger = LogManager.getLogger((String)"roguelike");
    private Coord origin;
    private final List<DungeonLevel> levels = new ArrayList<DungeonLevel>();
    private final WorldEditor editor;

    public Dungeon(WorldEditor editor) {
        this.editor = editor;
        try {
            RogueConfig.reload(false);
            SettingsResolver.getInstance(editor.getModLoader());
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    public static void generateInChunkIfPossible(WorldEditor editor, int chunkX, int chunkZ) {
        if (!Dungeon.isDungeonChunk(editor, chunkX, chunkZ)) {
            return;
        }
        logger.info("Trying to spawn dungeon at chunkX {} and chunkZ {}...", (Object)chunkX, (Object)chunkZ);
        Dungeon dungeon = new Dungeon(editor);
        Optional<Coord> coord = dungeon.selectLocation(editor.getRandom(), chunkX * 16, chunkZ * 16);
        if (!coord.isPresent()) {
            return;
        }
        Optional<DungeonSettings> settings = dungeon.getDungeonSettingsMaybe(coord.get());
        if (!settings.isPresent()) {
            return;
        }
        dungeon.timedGenerate(settings.get(), coord.get());
    }

    public static boolean isDungeonChunk(WorldEditor editor, int chunkX, int chunkZ) {
        return RogueConfig.DUNGEONS_SPAWN_ENABLED.getBoolean() && SpawnCriteria.isValidDimension(editor.getDimension()) && Dungeon.isSpawnFrequencyHit(chunkX, chunkZ) && Dungeon.isSpawnChanceHit(chunkX, chunkZ);
    }

    private static boolean isSpawnFrequencyHit(int chunkX, int chunkZ) {
        int frequency = Dungeon.getSpawnFrequency();
        return chunkX % frequency == 0 && chunkZ % frequency == 0;
    }

    private static int getSpawnFrequency() {
        return 3 * Math.max(2, RogueConfig.DUNGEONS_SPAWN_FREQUENCY.getInt());
    }

    private static boolean isSpawnChanceHit(int chunkX, int chunkZ) {
        double spawnChance = RogueConfig.DUNGEONS_SPAWN_CHANCE.getDouble();
        Random rand = new Random(Objects.hash(chunkX, chunkZ, 31));
        return (double)rand.nextFloat() < spawnChance;
    }

    public static int getLevel(int y) {
        if (y >= 45) {
            return 0;
        }
        if (y >= 35) {
            return 1;
        }
        if (y >= 25) {
            return 2;
        }
        if (y >= 15) {
            return 3;
        }
        return 4;
    }

    public void timedGenerate(DungeonSettings dungeonSettings, Coord coord) {
        new TimedTask("Dungeon.generate()", () -> this.generate(dungeonSettings, coord)).run();
    }

    public void generate(DungeonSettings dungeonSettings, Coord coord) {
        logger.info("Trying to spawn dungeon with id {} at {}...", (Object)dungeonSettings.getId(), (Object)coord);
        try {
            this.origin = coord.copy().setY(50);
            IntStream.range(0, dungeonSettings.getNumLevels()).mapToObj(dungeonSettings::getLevelSettings).map(DungeonLevel::new).forEach(this.levels::add);
            Arrays.stream(DungeonStage.values()).flatMap(stage -> DungeonTaskRegistry.getInstance().getTasks((DungeonStage)((Object)stage)).stream()).forEach(task -> this.performTaskSafely(dungeonSettings, (IDungeonTask)task));
            logger.info("Successfully generated dungeon with id {} at {}.", (Object)dungeonSettings.getId(), (Object)coord);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void performTaskSafely(DungeonSettings dungeonSettings, IDungeonTask task) {
        try {
            task.execute(this.editor, this, dungeonSettings);
        }
        catch (Exception exception) {
            new ReportThisIssueException(exception).printStackTrace();
        }
    }

    private Optional<Coord> selectLocation(Random rand, int x, int z) {
        int attempts = RogueConfig.DUNGEONS_SPAWN_ATTEMPTS.getInt();
        return IntStream.range(0, attempts).mapToObj(i -> Dungeon.getNearbyCoord(rand, x, z)).filter(this::canGenerateDungeonHere).findFirst();
    }

    private static Coord getNearbyCoord(Random random, int x, int z) {
        int distance = random.nextInt(Dungeon.getSpawnRadius());
        double angle = random.nextDouble() * 2.0 * Math.PI;
        int xOffset = (int)(Math.cos(angle) * (double)distance);
        int zOffset = (int)(Math.sin(angle) * (double)distance);
        return new Coord(x + xOffset, 0, z + zOffset);
    }

    private static int getSpawnRadius() {
        int spawnDiameter = Dungeon.getSpawnFrequency() * 16;
        return spawnDiameter / 2;
    }

    public boolean canGenerateDungeonHere(Coord coord) {
        Predicate<VanillaStructure> isTooCloseTo = structure -> this.hasStructureTooCloseBy(coord, (VanillaStructure)((Object)structure));
        Set<VanillaStructure> structuresToCheckDistanceTo = RogueConfig.vanillaStructuresToCheckDistanceTo();
        if (!structuresToCheckDistanceTo.isEmpty() && structuresToCheckDistanceTo.stream().anyMatch(isTooCloseTo)) {
            return false;
        }
        Coord cursor = coord.copy().setY(RogueConfig.UPPERLIMIT.getInt());
        return this.editor.isAirBlock(cursor) && this.canFindStartingCoord(cursor) && this.isFreeOverhead(cursor) && this.isSolidBelow(cursor);
    }

    private boolean hasStructureTooCloseBy(Coord coord, VanillaStructure structure) {
        int minimumDistanceRequired = RogueConfig.SPAWN_MINIMUM_DISTANCE_FROM_VANILLA_STRUCTURES.getInt();
        Coord structureCoord = this.editor.findNearestStructure(structure, coord, minimumDistanceRequired);
        if (structureCoord == null) {
            logger.info("Did not detect structure \"{}\" within {} blocks of potential spawn location {}.", (Object)structure.name(), (Object)minimumDistanceRequired, (Object)coord);
            return false;
        }
        return coord.distance(structureCoord) < (double)minimumDistanceRequired;
    }

    private boolean canFindStartingCoord(Coord cursor) {
        while (!this.editor.isValidGroundBlock(cursor)) {
            cursor.down();
            if (cursor.getY() < RogueConfig.LOWERLIMIT.getInt()) {
                return false;
            }
            if (!this.editor.isBlockOfTypeAt(BlockType.WATER_STILL, cursor)) continue;
            return false;
        }
        return true;
    }

    private boolean isFreeOverhead(Coord cursor) {
        Coord start = cursor.copy().up(4).translate(new Coord(-4, 0, -4));
        Coord end = cursor.copy().up(4).translate(new Coord(4, 0, 4));
        for (Coord c : RectSolid.newRect(start, end)) {
            if (!this.editor.isValidGroundBlock(c)) continue;
            return false;
        }
        return true;
    }

    private boolean isSolidBelow(Coord cursor) {
        Coord start1 = cursor.copy().translate(new Coord(-4, -3, -4));
        Coord end1 = cursor.copy().translate(new Coord(4, -3, 4));
        int airCount = 0;
        for (Coord c : RectSolid.newRect(start1, end1)) {
            if (!this.editor.isValidGroundBlock(c)) {
                ++airCount;
            }
            if (airCount <= 8) continue;
            return false;
        }
        return true;
    }

    private Optional<DungeonSettings> getDungeonSettingsMaybe(Coord coord) {
        if (RogueConfig.RANDOM.getBoolean()) {
            return Optional.of(new SettingsRandom(this.editor.getRandom()));
        }
        return SettingsResolver.getInstance(this.editor.getModLoader()).chooseRandom(this.editor, coord);
    }

    public Coord getPosition() {
        return this.origin.copy();
    }

    public List<DungeonLevel> getLevels() {
        return this.levels;
    }

    public void generateLayout(WorldEditor editor) {
        Coord start = this.getPosition();
        for (DungeonLevel level : this.getLevels()) {
            start = level.generateLayout(editor, start).down(10);
        }
    }

    public void encase(WorldEditor editor) {
        if (RogueConfig.ENCASE.getBoolean()) {
            this.getLevels().forEach(level -> level.filter(editor, Filter.get(Filter.ENCASE)));
        }
    }

    public void tunnel(WorldEditor editor) {
        this.getLevels().forEach(level -> level.tunnel(editor));
    }

    public void generateSegments(WorldEditor editor) {
        this.getLevels().forEach(level -> level.generateSegments(editor));
    }

    public void generateRooms() {
        this.getLevels().forEach(DungeonLevel::generateRooms);
    }

    public void linkLevels(WorldEditor editor) {
        this.getLevels().stream().reduce(null, (prev, level) -> level.generateLinkers(editor, (DungeonLevel)prev));
    }

    public void generateTower(WorldEditor editor, DungeonSettings settings) {
        TowerType tower = settings.getTower().getType();
        Theme theme = settings.getTower().getTheme();
        Coord at = this.getPosition();
        tower.instantiate(editor, theme).generate(at);
    }

    public void applyFilters(WorldEditor editor) {
        this.getLevels().forEach(level -> level.applyFilters(editor));
    }
}

