import { getUserId } from "./customAuthFlow.js";
import { getBearer } from "./customBearer.js";
import { customFetch } from "./customFetch.js";
import {
  createSchool,
  removeSchool,
  startGame,
} from "./customGame.js";
import { createNotification } from "./customNotification.js";

// TABLE

async function getTable(id) {
  let response = await customFetch(
    `api/Tables/${id}`,
    "GET",
    null,
    getBearer()
  );

  if (!response.ok) {
    tableNotFound(id);
  } else {
    let table = response.data;
    return table;
  }
}

function tableNotFound(id) {
  window.location.href = `/Frontend/src/auth/tables/join.html?notification=Table with id "${id}" not found`;
}

async function getAvailableTables() {
  let response = await customFetch(
    "api/Tables/with-available-seats",
    "GET",
    null,
    getBearer()
  );

  if (!response.ok) {
    createNotification(response.status);
  } else {
    let availableTables = document.getElementById("availableTables");
    availableTables.replaceChildren();

    if (response.data.length > 0) {
      response.data.forEach((table) => {
        let div1 = document.createElement("div");
        let div2 = document.createElement("div");
        let h1 = document.createElement("h1");
        let div3 = document.createElement("div");
        let a = document.createElement("a");
        let i = document.createElement("i");

        table.seatedPlayers.forEach((player) => {
          let p = document.createElement("p");

          if (player.id == table.ownerPlayerId) {
            let i = document.createElement("i");
            i.classList.add("fa-brands", "fa-square-web-awesome-stroke");
            i.style.color = "gold";
            p.appendChild(i);
            p.classList.add("flex", "items-center", "gap-1");
          }

          p.appendChild(document.createTextNode(player.name));
          p.classList.add(`text-${player.color}-500`);

          div3.appendChild(p);
        });

        div3.classList.add(
          "flex",
          "gap-4",
          "overflow-x-scroll",
          "no-scrollbar",
          "text-lg",
          "lg:text-xl"
        );
        h1.appendChild(document.createTextNode(table.id));
        h1.classList.add("text-lg", "lg:text-3xl", "border-b-2");

        div2.appendChild(h1);
        div2.appendChild(div3);

        div2.classList.add("font-mono", "flex", "flex-col", "grow", "gap-2");
        a.classList.add(
          "text-lg",
          "lg:text-3xl",
          "p-3",
          "hover:text-yellow-500",
          "transition-all",
          "duration-300",
          "flex",
          "justify-center",
          "items-center",
          "gap-2"
        );
        a.appendChild(document.createTextNode("JOIN"));
        i.classList.add("fa-solid", "fa-arrow-right");
        a.appendChild(i);
        a.href = `/Frontend/src/auth/tables/table.html?tableId=${table.id}`;

        div1.appendChild(div2);
        div1.appendChild(a);

        div1.classList.add(
          "w-full",
          "flex",
          "flex-col",
          "lg:flex-row",
          "p-4",
          "lg:p-8",
          "gap-2"
        );

        availableTables.appendChild(div1);
      });
    } else {
      let div = document.createElement("div");
      div.classList.add(
        "w-full",
        "text-xl",
        "lg:text-3xl",
        "flex",
        "justify-center",
        "items-center",
        "gap-4",
        "p-4"
      );
      div.appendChild(document.createTextNode("No tables available"));
      let i = document.createElement("i")
      i.classList.add("fa-solid", "fa-sync")
      i.addEventListener("click", () => getAvailableTables())
      div.appendChild(i);
      availableTables.appendChild(div);
    }
  }
}

async function createTable(numberOfPlayers, playmatSize, movecardSet) {
  let body = {
    numberOfPlayers: 2,
    playerMatSize: 5,
    moveCardSet: 0,
  };

  let response = await customFetch("api/Tables", "POST", body, getBearer());

  if (!response.ok) {
    createNotification(response.status);
  } else {
    window.location.href = `/Frontend/src/auth/tables/table.html?tableId=${response.data.id}`;
  }
}

// TABLE MECHANICS

async function joinTable(id) {
  let table = await getTable(id);
  if (table.seatedPlayers.some((player) => player.id == getUserId())) {
    createNotification(`Welcome to table ${table.id}`);
  } else if (!table.hasAvailableSeat) {
    window.location.href =
      "/Frontend/src/auth/lobby.html?notification=Table has no available seats";
  } else {
    let response = await customFetch(
      `api/Tables/${id}/join`,
      "POST",
      null,
      getBearer()
    );

    if (!response.ok) {
      createNotification(response.status);
    } else {
      window.location.reload();
    }
  }
}

async function leaveTable(id) {
  let response = await customFetch(
    `api/Tables/${id}/leave`,
    "POST",
    null,
    getBearer()
  );

  if (!response.ok) {
    createNotification(response.message);
  } else {
    window.location.href =
      "/Frontend/src/auth/lobby.html?notification=Left table succesfully";
  }
}

let previousSeatedPlayers = [];
let previousOwner;

async function displayUsers(tableId) {
  let table = await getTable(tableId);
  if (table.gameId != '00000000-0000-0000-0000-000000000000'){
    window.location.href = `/Frontend/src/auth/games/game.html?gameId=${table.gameId}`;
  }

  // Check if players have left
  previousSeatedPlayers.forEach((previousPlayer) => {
    if (!table.seatedPlayers.some((player) => player.id == previousPlayer.id)) {
      playerHasLeft(previousPlayer);
    }
  });

  // Check if players have joined
  table.seatedPlayers.forEach((seatedPlayer) => {
    if (!previousSeatedPlayers.some((player) => player.id == seatedPlayer.id)) {
      playerHasJoined(seatedPlayer, table);
    }
  });

  // Check user owner
  if (table.ownerPlayerId != previousOwner) {
    table.seatedPlayers.forEach((player) => {
      if (player.id == table.ownerPlayerId) {
        let element;
        if (isCurrentPlayer(player.id)) {
          element = document
            .getElementById(`currentPlayer`)
            .querySelector("h1");
        } else {
          element = document.getElementById(`enemyPlayer`).querySelector("h1");
        }

        setOwner(player, element);
      }
    });

    // Set start button for owner
    if (table.ownerPlayerId == getUserId()) {
      setStartButton(table.id);
    }
    previousOwner = table.ownerPlayerId;
  }

  previousSeatedPlayers = table.seatedPlayers;

  setTimeout(() => {
    displayUsers(table.id);
  }, 5000);
}

function isCurrentPlayer(playerId) {
  return playerId == getUserId();
}

function playerHasLeft(player) {
  let element;
  if (isCurrentPlayer(player.id)) {
    element = document.getElementById(`currentPlayer`).querySelector("h1");
  } else {
    element = document.getElementById(`enemyPlayer`).querySelector("h1");
  }
  removeSchool(player);
  resetSpinner(element);
  createNotification(`Player ${player.name} has left`);
}

function playerHasJoined(player, table) {
  let usernameDisplay;
  if (isCurrentPlayer(player.id)) {
    usernameDisplay = document.getElementById(`currentPlayer`);
  } else {
    usernameDisplay = document.getElementById(`enemyPlayer`);
    createNotification(`Player ${player.name} has joined`);
  }

  let h1 = usernameDisplay.querySelector("h1");

  h1.replaceChildren();

  if (player.id == table.ownedPlayerId) {
    let i = document.createElement("i");
    i.classList.add("fa-brands", "fa-square-web-awesome-stroke");
    i.style.color = "gold";
    h1.appendChild(i);
  }

  h1.appendChild(document.createTextNode(player.name));

  createSchool(player);
}

async function getDirectionForPlayer(tableId) {
  let response = await customFetch(
    `api/Tables/${tableId}`,
    "GET",
    null,
    getBearer()
  );

  if (!response.ok) {
    tableNotFound(id);
  } else {
    let table = response.data;
    const player = table.seatedPlayers.find(
      (player) => player.id == getUserId()
    );
    if (player) {
      return player.direction;
    }
    return "North";
  }
}

function resetSpinner(element) {
  // Remove existing text
  element.replaceChildren();

  // Waiting text
  element.appendChild(document.createTextNode("Waiting for player"));

  // Spinner icon
  let spinner = document.createElement("i");
  spinner.classList.add("fa-solid", "fa-sync", "fa-spin");

  element.appendChild(spinner);
}

function setOwner(player, element) {
  // Remove existing text
  element.replaceChildren();

  // Crown icon
  let i = document.createElement("i");
  i.classList.add("fa-brands", "fa-square-web-awesome-stroke");
  i.style.color = "gold";
  element.appendChild(i);

  // Player name
  element.appendChild(document.createTextNode(player.name));
}

function setStartButton(tableId) {
  let button = document.getElementById("startGameButton");
  button.replaceChildren();

  button.appendChild(document.createTextNode("START"));

  let arrowIcon = document.createElement("i");
  arrowIcon.classList.add("fa-solid", "fa-arrow-right");
  button.appendChild(arrowIcon);

  button.addEventListener("click", () => startGame(tableId));
}

export {
  getTable,
  tableNotFound,
  createTable,
  getAvailableTables,
  joinTable,
  leaveTable,
  getDirectionForPlayer,
  displayUsers,
};
